package myposhbot;

import cz.cuni.astar.AStarResult;
import cz.cuni.pogamut.Client.GameMap;
import cz.cuni.pogamut.Client.PathManager;
import cz.cuni.pogamut.Client.PathTypes;
import cz.cuni.pogamut.MessageObjects.Item;
import cz.cuni.pogamut.MessageObjects.NavPoint;
import cz.cuni.pogamut.MessageObjects.Path;
import cz.cuni.pogamut.MessageObjects.Player;
import cz.cuni.pogamut.MessageObjects.Triple;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Set;

/**
 * This class encapsulate work with paths during navigation
 * so it is on one place and it is clearer what is going on with them
 * <br>
 * It takes care about restarting when target is changed etc.
 * It is useful when agent changes targets and would like to look consistent 
 * in doing so.
 * <p>
 * So the way how to use it:
 * <ul>
 * <li> Call checkPath() to know whether the path is ready - e.g. properly initialized
 * <li> If not, call preparePath() - that will prepare everything - initialize the path
 * <li> and after that, you can call runAlongPath with no problems
 * </p>
 * @author Ondra
 */

// co bych dal za to, kdyby se vsude pouzival jen interface :))
public class MyPathManager extends PathManager {    
    
    /** structure that stores collected paths */
    protected HashMap<Integer, ArrayList<NavPoint>> collectedPaths = new HashMap<Integer, ArrayList<NavPoint>>();
    /** agent than knows which way he has to go on the lift (mover) */
    protected boolean isUpOnTheLift = false;
    /** path to next item - array of nav points */
    protected ArrayList<NavPoint> pathToRunAlong = null;
    /** restart of the actual path. is on when something changes - usualy target - and triggers restart*/
    protected boolean restart = false;
    /** this structure is to keep information about the readiness of the paths of different types */
    protected HashMap<PathTypes, Boolean> pathReady = new HashMap<PathTypes, Boolean>(); 
    /** this structure is to keep information about the targets of the paths of different types */
    protected HashMap<PathTypes, Object> targets = new HashMap<PathTypes, Object>();
    /** actual path type */
    protected PathTypes actualPathType = null;
    /** back pointer to gameMap*/
    protected GameMap gameMap = null;
    /** counter of walking attempts - to prevent stuck */
    public int walking = 0;
    /** index of the next navigation point of the path */
    public int nextNavPointIndex = 0;
    
    /**
     * path is not ready in next cases: 
     * <ol> 
     *  <li> if the type of the path has changed
     *  <li> if the target is different than the stored one 
     * </ol>
     * then it returns the value of readiness of corresponding type
     */
    @Override
    public boolean checkPath(PathTypes type, Object target) {
        if (!type.equals(actualPathType)) { // if the types of target are not equal, than agent changed the type of path
                                            // and so it is not ready and has to be restarted
            pathReady.put(type, false);
            pathReady.put(actualPathType, false);
            actualPathType = type;
            restart = true;
        }
        if (!target.equals(targets.get(type))) { // if the target is different it has to be restarted - last setting is for obsolete target
            pathReady.put(type, false);
            targets.put(type, target);
            restart = true;
        }
        return pathReady.get(type);
    }
    /** sets every readiness to false */
    @Override
    public void restartPathReady() {
        Set<PathTypes> keySet = pathReady.keySet();
        for (PathTypes al : keySet)
            pathReady.put(al, false);
    }
    /** restarts all path variables - almost all variables from Path manager */
    @Override
    protected void restartPathVariables() {
            restart = false;
            pathToRunAlong = null;
            isUpOnTheLift = false;
            pathRequestSent = false;
            collectedPaths = new HashMap<Integer, ArrayList<NavPoint>>(); // this is vital to not receive some path from obsolete starting position
            nextNavPointIndex = 0;
            walking = 0;
            isUpOnTheLift = false;        
    }
    /**
     * preparePath is the most important method of PathManager
     * it is responsible for obtaining path according to the PathType and according to useAStar
     */
    @Override
    public void preparePath(PathTypes type, Object target, boolean useAStar) {
        // if path was not restarted yet, restart everything
        if (restart)
            restartPathVariables();
        boolean result = false;
        switch (type) {
            case ITEM:
                if (useAStar && (((Item) target).navPoint != null) && (((Item) target).navPoint.UnrealID != null))
                    result = obtainPathAStar(((Item) target).navPoint);
                else
                    result = obtainPathGB(((Item) target).location);
                break;
            case LOCATION:
                result = obtainPathGB((Triple) target);
                break;
            case PLAYER:
                result = obtainPathGB(((Player) target).location);
                break;
            case NAVPOINT:
                if (useAStar)
                    result = obtainPathAStar((NavPoint) target);
                else
                    result = obtainPathGB(((NavPoint) target).location);
                break;
            default:
                result = obtainPathGB((Triple) target);
                break;
        }
        if (result)
            pathReady.put(type, true);
    }

    /**
     * Obtain path from gamebots
     */    
    protected int counterPathID = 10000;
    protected int pathID = 0;
    protected boolean pathRequestSent = false;
    
    /**
     * obtain path to specified location keeps returning null until path is received
     * don't forget to initialize it - initializeGetPath should be called prior this method
     * @param location
     * @return null if it has not received path from Gamebots yet, path if it does
     */
    @Override
    protected ArrayList<NavPoint> getPathToLocation(Triple location) {
        if (pathRequestSent) {
            if (collectedPaths.get(pathID) == null)
                return null;
            return collectedPaths.get(pathID);
        }
        ++counterPathID;
        if (counterPathID == Integer.MAX_VALUE)
                counterPathID = 10000;				
        pathID = counterPathID;
        this.gameMap.sendGetPathToLocation(pathID, location);
        pathRequestSent = true;
        return null;
    }
            
    /**
     * obtain path from GB
     */
    @Override
    protected boolean obtainPathGB (Triple location) {
        pathToRunAlong = getPathToLocation(location); 
        if (pathToRunAlong == null)
            return false;
        return true;
    }
    /**
     * obtain path from built-in A* - only suitable for Navigation Points
     */
    @Override
    protected boolean obtainPathAStar (NavPoint target) {
        AStarResult result = gameMap.getPathAStar(target);
        if (result.success) {
            this.pathToRunAlong = gameMap.getNavPointsAStar(result);
            return true;
        } else {
            if (!obtainPathGB(target.location))
                return false;
            else
                return true;
        }
    }
    
    /** Creates a new instance of PathManager */
    public MyPathManager(GameMap gameMap) {
        super(gameMap);
        this.gameMap = gameMap;
    }
    /**
     * returns current navigation point of the current path to which agent should run to
     */
    @Override
    public NavPoint getCurrentNavPointOfPath() {
        if (nextNavPointIndex < pathToRunAlong.size())
            return pathToRunAlong.get(nextNavPointIndex);
        return null;
    }
    /**
     * returns next navigation point of the current path
     */
    @Override
    public NavPoint getNextNavPointOfPath() {
        if ((nextNavPointIndex + 1) < pathToRunAlong.size())
            return pathToRunAlong.get(nextNavPointIndex + 1);
        return null;
    }
    /**
     * stores new path to collectedPaths
     */
    @Override
    protected void newGBPathReceived(Path path) {
        this.collectedPaths.put(Integer.valueOf(path.pongID), path.nodes);
    }

    /** 
     * function for compatibility with someone trying to do everything by himself...and therefore using 
     * older API
     *
     * it initializes all variables which has to be initialized prior the call of the method runAlongThePath
     */
    void initializeRunAlongPathManually(ArrayList<NavPoint> path) {
        this.pathToRunAlong = path;
        this.walking = 0;
        this.isUpOnTheLift = false;
        this.pathRequestSent = false;
        this.nextNavPointIndex = 0;
    }
    
    
}
